// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Remix_RGBA.B.fsh                               ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.0 (2017/04/04)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To modify the colours of an image by remixing the red, green, ==
// == blue and alpha components of each pixel with the pixels to be modified ==
// == selected by their colour using a very simple R,G,B level based method. ==
// ==                                                                        ==
// == Description: The shader is a more powerful version of Emu_Remix_RGBA.A ==
// == but uses more GPU resources. It performs the same colour mixing as the ==
// == simpler shader but only does it for pixels that pass a  simple  colour ==
// == colour selection procedure.  The Red, Green, Blue and Alpha components ==
// == of source image pixels that meet the selection criteria  are  modified ==
// == using                                                                  ==
// ==                                                                        ==
// ==    C.r' = C.r + K.r + C.r*R.r + C.g*R.g + C.b*R.b + C.a*R.a            ==
// ==    C.g' = C.g + K.g + C.r*G.r + C.g*G.g + C.b*G.b + C.a*G.a            ==
// ==    C.b' = C.b + K.b + C.r*B.r + C.g*B.g + C.b*B.b + C.a*B.a            ==
// ==    C.a' = C.a + K.a + C.r*A.r + C.g*A.g + C.b*A.b + C.a*A.a            ==
// ==                                                                        ==
// == or more succinctly                                                     ==
// ==                                                                        ==
// ==    C.r' = C.r + K.r + dot(C,R)                                         ==
// ==    C.g' = C.g + K.g + dot(C,G)                                         ==
// ==    C.b' = C.b + K.b + dot(C,B)                                         ==
// ==    C.a' = C.a + K.a + dot(C,A)                                         ==
// ==                                                                        ==
// == or even more succinctly as                                             ==
// ==                                                                        ==
// ==    C' = C + K + C*M                                                    ==
// ==                                                                        ==
// == where C is the source colour, R, G, B, A, K four element vectors and M ==
// == the 4x4 matrix with R, G, B and A as its rows.                         ==
// ==                                                                        ==
// == The parameters K, R, G, B and A are supplied as uniform variables that ==
// == may be defined in .scn files using the shader. All default to zero.    ==
// ==                                                                        ==
// == The colour mixing is only performed if the initial colour satisfies  a ==
// == selection criteria based on a set of low and high limits for  each  of ==
// == the R, G, B and A components.  In the simplest case these specify a 4D ==
// == cube in the RGBA colour space and the selection criteria are           ==
// ==                                                                        ==
// ==    C0.r <= C.r <= C1.r                                                 ==
// ==    C0.g <= C.g <= C1.g                                                 ==
// ==    C0.b <= C.b <= C1.b                                                 ==
// ==    C0.a <= C.a <= C1.a                                                 ==
// ==                                                                        ==
// == which will select any point inside the 4D cube specified by C0 and C1. ==
// == Alternatively,  the R, G, B components of C0 and C1 may be regarded as ==
// == specifying a 3D elipsoid that fits inside the 3D RGB cuboid, an option ==
// == is provided to use this interpretation and the selection criteria then ==
// == are based on the RGB distances as fractions of the elipsoid's radii as ==
// == calculated using                                                       ==
// ==                                                                        ==
// ==    C2.rgb = ( C0.rgb + C1.rgb ) / 2   // Centre of the elipsoid        ==
// ==    RR.rgb = ( C1.rgb - C0.rgb ) / 2   // Radii of the elipsoid         ==
// ==    DD.rgb = abs ( c - C2 ) / RR       // Fractional RGB distances      ==
// ==                                                                        ==
// == with the selection criteria then being                                 ==
// ==                                                                        ==
// ==    length(DD) <= 1.0                                                   ==
// ==    C0.a <= C.a <= C1.a                                                 ==
// ==                                                                        ==
// == A minor variation using the Manhattan distance measure in place of the ==
// == normal Euclidian length is also provided. If Manhattan distances, i.e. ==
// == sum of R, G, B distances rather than sum of squares, are used then the ==
// == elipsoid reduces to an octahedon, i.e. to a bi-pyramid.  The elipsoid, ==
// == octahedron or cube may extend beyond the normal 0.0 to 1.0 limits used == 
// == for R, G and B,  most usefully by placing the center of an elipsoid or ==
// == octahedron on a side of the colour space to select a cup or  pyramidal ==
// == region.                                                                ==  
// ==                                                                        ==
// == Note, the colour selection mechanism is a very simple one but does not ==
// == work well with most natuarly coloured images.  It can, for example, be ==
// == used to select pure red areas by requiring both a high red  level  and ==
// == low blue and green levels.  But in naturaly coloured images a red area ==
// == is likely to have significant blue or green components as well as red, ==
// == and including these when trying to select the whole of the red area is ==
// == likely either to select areas that aren't perceived as red or will not ==
// == select all of the red area.  The  elipsoid or octahedral variations of ==
// == the selection method can help, but usualy not by much because the have ==
// == fixed orientaions in colour space that are rarely what is wanted. Much ==
// == better colour selection should be possible using an HSV or  HSL  (hue, ==
// == saturation and value or luminosity) colour space and it is intended to ==
// == to provide shaders that use these for colour selection.                ==
// ==                                                                        ==
// == The shader can be used to achieve several colour effects. e.g.         ==
// ==                                                                        ==
// ==   Remove one or more colour components                                 ==
// ==   Invert one or more colour components                                 ==
// ==   Reduce or enhance one or more components                             ==
// ==   Average over two or more colour components                           ==
// ==   Rotate about the luminosity axis in colour space.                    ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/04/04 - v1.0.0 - Initial version (from Emu_Remix_RGBA.B.fsh)    ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

vec2 Emu_Normalise_to_Window ( vec2 xy ) { return xy / u_WindowSize.xy; }

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Boolean_Vector_Ops.lib.src

#define Emu_bvec4_or(A,B)  bvec4 ( A[0]||B[0], A[1]||B[1], A[2]||B[2], A[3]||B[3] )
#define Emu_bvec4_and(A,B) bvec4 ( A[0]&&B[0], A[1]&&B[1], A[2]&&B[2], A[3]&&B[3] )
#define Emu_bvec4_xor(A,B) bvec4 ( A[0]^^B[0], A[1]^^B[1], A[2]^^B[2], A[3]^^B[3] )

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// EmuLib standard scale and hotspot parameters.

uniform vec2 Emu_Remix_RGBA_scale;
uniform vec2 Emu_Remix_RGBA_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Remix_RGBA_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Remix_RGBA_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the output of the shader is calculated as described above  with
// no attempt being made to ensure that the calculated colour components  are
// within their valid range of 0.0 to 1.0.  The mode control parameter can be
// used to modify the result to ensure that it is valid and to  select  which
// colour selection criteria is used.  It  is a three digit integer where the
// sign and most significant and digit control colour selection,  the  middle
// digit controls colour and opacity clamping and the least significant digit
// controls luminosity preservation.
// 
// Valid values for the colour selection mode are
//
//   0 - default, equivalent to 1
//   1 - basic RGBA hypercube condition
//   2 - RGB elipsoid condition using normal distance
//   3 - RGB octahedal condition using Manhattan distance
//   6 - as 1 but with inverted opacity condition
//   7 - as 2 but with inverted opacity condition
//   8 - as 3 but with inverted opacity condition
//
// negating the mode inverts the selection.
//
// Valid values for the clamp mode, i.e. the middle digit, are
//
//   0 - default - no clamping
//   1 - clamp R, G, B and A
//   2 - clamp R, G and B
//   3 - clamp A only
//
// and those for the luminosity mode, the least significant digit, are
//
//   0 - default - no luminosity correction
//   1 - luminosity preserved

uniform int Emu_Remix_RGBA_mode;

int selection_mode  = int ( mod(Emu_Remix_RGBA_mode/100.0,10.0) );
int clamping_mode   = int ( mod(Emu_Remix_RGBA_mode/10.0, 10.0) );
int luminosity_mode = int ( mod(Emu_Remix_RGBA_mode,      10.0) );

bool invert_selection   = Emu_Remix_RGBA_mode < 0;
bool invert_select_A    = selection_mode >= 5;
int  RGB_selection_mode = int(max(mod(float(selection_mode),5.0),1.0));

bool constant_luminosity = luminosity_mode == 1;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The selection parameters. These, as explained above specify the range of
// initial colours that will be processed.  Any pixel with a colour outside
// the specified range will retain its initial colour. The defaults for the
// limits are 0.0 and 1.0.  Values outside the normal valid range of 0.0 to
// 1.0 for colour components may be specified for these parameters, e.g. to
// position the center of a selection elipsoid close to the edge of the RGB
// colour space or to allow the shader to process the output  from  another
// shader that may deliberately produce colour components that are  not  in 
// the normal range.

uniform vec2 Emu_Remix_RGBA_Cr; // Limits for the R component
uniform vec2 Emu_Remix_RGBA_Cg; // Limits for the G component
uniform vec2 Emu_Remix_RGBA_Cb; // Limits for the B component
uniform vec2 Emu_Remix_RGBA_Ca; // Limits for the A component

vec2 Cr = EMU_DEFAULT ( vec2, Emu_Remix_RGBA_Cr, vec2(0.0,1.0) );
vec2 Cg = EMU_DEFAULT ( vec2, Emu_Remix_RGBA_Cg, vec2(0.0,1.0) );
vec2 Cb = EMU_DEFAULT ( vec2, Emu_Remix_RGBA_Cb, vec2(0.0,1.0) );
vec2 Ca = EMU_DEFAULT ( vec2, Emu_Remix_RGBA_Ca, vec2(0.0,1.0) );

vec4[2] CC = vec4[2] ( vec4 ( Cr[0], Cg[0], Cb[0], Ca[0] ),
                       vec4 ( Cr[1], Cg[1], Cb[1], Ca[1] )
                     );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The colour mixing parameters.  See the overall description of this shader,
// above, for details of what each of these are.

uniform vec4 Emu_Remix_RGBA_R; // Mixing parameters for the Red channel
uniform vec4 Emu_Remix_RGBA_G; // Mixing parameters for the Green channel
uniform vec4 Emu_Remix_RGBA_B; // Mixing parameters for the Blue channel
uniform vec4 Emu_Remix_RGBA_A; // Mixing parameters for the Alpha channel
uniform vec4 Emu_Remix_RGBA_K; // Additive terms for the R,G,B,A channels

#define R Emu_Remix_RGBA_R
#define G Emu_Remix_RGBA_G
#define B Emu_Remix_RGBA_B
#define A Emu_Remix_RGBA_A
#define K Emu_Remix_RGBA_K

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Determine the lower and upper limits to which the colour channels will be
// clamped.  If clamping is not enabled then the limits used are effectively
// infinite.

bool clamp_RGB = ( clamping_mode == 1 ) || ( clamping_mode == 2 );
bool clamp_A   = ( clamping_mode == 1 ) || ( clamping_mode == 3 );
vec4 clamp_X   = vec4 ( vec3(float(clamp_RGB)), clamp_A );

vec4 clamp_low  = -1.0e32 * clamp_X;
vec4 clamp_high = 1.0 - clamp_low;

// ============================================================================
// == The shader's major function =============================================
// ============================================================================

vec4 Emu_Remix_RGBA ( vec2 uv )
 {
   // Get and manipulate the colour and opacity for the current point.

   vec4 c = texture2D ( iChannel0, uv );

   // Determine if colour is in the selected range.

   bool selected;

   switch ( RGB_selection_mode )
    { 
      vec3 C2, RR, DD;

      case 1:
         selected = all(lessThanEqual(CC[0].rgb,c.rgb))
                 && all(lessThanEqual(c.rgb,CC[1].rgb));
         break;

      case 2: // Simple RGB elipsoid criteria
         C2.rgb = ( CC[1].rgb + CC[0].rgb ) * 0.5; // Centre of the elipsoid
         RR.rgb = ( CC[1].rgb - CC[0].rgb ) * 0.5; // Size of the elipsoid
         DD.rgb = abs ( c.rgb - C2 ) / RR;         // Fractional distances
         selected = length(DD) <= 1.0;             // Use Eulerian distance
         break;

      case 3: // Manhattan distance criteria
         C2.rgb = ( CC[1].rgb + CC[0].rgb ) * 0.5; // Centre of the octahedron
         RR.rgb = ( CC[1].rgb - CC[0].rgb ) * 0.5; // Size of the octahedron
         DD.rgb = abs ( c.rgb - C2 ) / RR;         // Fractional distances
         selected = dot(DD,vec3(1.0)) <= 1.0;      // Use Manhattan distance
         break;
  
    }

   bool select_A = ( CC[0].a <= c.a ) && ( c.a <= CC[1].a );

   select_A = select_A ^^ invert_select_A;
   selected = selected && select_A;
   selected = selected ^^ invert_selection;

   // Return original colour if outside the selected range.

   if ( ! selected ) return c;

   // Original colour was in range so process this pixel.

   float old_luminosity = length(c.rgb);

   c = c + K + c*mat4 ( R, G, B, A );

   // Optionaly ensure that luminosity is preserved.

   if ( constant_luminosity )
    { c.rgb = c.rgb * old_luminosity / length(c.rgb);
    }

   // Clamp the result - if no clamping was requested then the
   // limits used here are effectively infinite.

   return clamp ( c, clamp_low, clamp_high );

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   uv = ( uv - hotspot ) / scale + hotspot;

   // Perform the mixing and update the shader's outputs.

   gl_FragColor = Emu_Remix_RGBA(uv) * gl_Color;

}